/**
 * @license
 * Copyright 2025 Elron
 * SPDX-License-Identifier: GPL3
 */

/**
 * @fileoverview different sensors blocks for Blockly
 * @author info@elron.tech
 */
 
'use strict';

goog.require('Blockly.Arduino');

// ================================= DHT ================================= //
Blockly.Arduino['dht_block'] = function() {
  var pin  = this.getFieldValue('PIN');
  var type   = this.getFieldValue('TYPE');
  var value   = this.getFieldValue('VALUE');
  var instance = 'dht_' + pin;

  Blockly.Arduino.includes_['dht_include'] = '#include <DHT.h>';
  Blockly.Arduino.definitions_['def_'+ instance] = 'DHT ' + instance + '(' + pin +', ' + type + ');';
  Blockly.Arduino.setups_['setup_' + instance] = instance + '.begin();';
  
  var code = '(int)(0.5f + ' + instance + (value === 'temp' ? '.readTemperature())' : '.readHumidity())');
  return [code, Blockly.Arduino.ORDER_ATOMIC];
};

// ================================= HCSR04 ================================= //
Blockly.Arduino['hcsr04_block'] = function() {
  var trig_pin  = this.getFieldValue('TRIG');
  var echo_pin   = this.getFieldValue('ECHO');
  var instance = 'hcsr04_' + trig_pin + '_' + echo_pin;

  Blockly.Arduino.includes_['hcsr04_include'] = '#include <HCSR04.h>';
  Blockly.Arduino.definitions_['def_'+ instance] = 'UltraSonicDistanceSensor ' + instance + '(' + trig_pin +', ' + echo_pin + ');';
  
  var code = '(int)(0.5f + ' + instance + '.measureDistanceCm())';
  return [code, Blockly.Arduino.ORDER_ATOMIC];
};

// ================================= RFID ================================= //
Blockly.Arduino['rfid_get_uid'] = function(block) {
  const ss = block.getFieldValue('SS');
  const rst = block.getFieldValue('RST');
  const instance = 'rfid_' + ss + '_' + rst;

  Blockly.Arduino.includes_['spi']      = '#include <SPI.h>';
  Blockly.Arduino.includes_['mfrc522']  = '#include <MFRC522.h>';
  Blockly.Arduino.definitions_['def_mfrc522_' + instance] = 'MFRC522 ' + instance + '(' + ss + ', ' + rst + ');';
  Blockly.Arduino.setups_['setup_mfrc522_' + instance]    = 'SPI.begin();\n  ' + instance + '.PCD_Init();';

  // Добавляем функцию чтения номера в строку
  Blockly.Arduino.userFunctions_['rfid_read_uid'] = `
String readUid(MFRC522 &reader) {
  String uid = "";
  if (reader.PICC_IsNewCardPresent() && reader.PICC_ReadCardSerial()) {
    for (byte i = 0; i < reader.uid.size; i++) {
      uid += (String(reader.uid.uidByte[i], HEX) + String(" "));
    }
    reader.PICC_HaltA();
  }
  return uid;
}`;

  // Возвращаем вызов с нужным экземпляром
  return ['readUid(' + instance + ')', Blockly.Arduino.ORDER_ATOMIC];
};

// ================================= DS18B20 ================================= //
Blockly.Arduino['ds18b20_block'] = function() {
  var pin  = this.getFieldValue('PIN');
  var instance = 'ds18b20_' + pin;

  Blockly.Arduino.includes_['ds18b20_include'] = '#include <DS18B20.h>';
  Blockly.Arduino.definitions_['def_'+ instance] = 'DS18B20 ' + instance + '(' + pin + ');';

  var code = '(int)(0.5f + ' + instance + '.getTempC())';

  return [code, Blockly.Arduino.ORDER_ATOMIC];
}